<?php

namespace JLGR\Views\Concerns;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Facades\Auth;
use JLGR\Views\Models\View;
use JLGR\Views\Models\ViewStatistic;

trait HasViews
{

	/**
	 * Get all views for the model.
	 */
	public function views () : MorphMany
	{
		return $this->morphMany(View::class, 'viewable')->latest();
	}

	/**
	 * Get statistics for this model.
	 */
	public function viewStatistics () : MorphOne
	{
		return $this->morphOne(ViewStatistic::class, 'viewable');
	}

	/**
	 * Register view for this model.
	 * 
	 * @param  integer  $userId
	 * 
	 * @return \JLGR\Views\Models\View
	 */
	public function registerView (?int $userId = null) : View
	{
		$userId = $userId ?? Auth::id();
		$isUnique = false;
		
		// Check if this is the first view for the user.
		if ($userId && !$this->views()->where('user_id', $userId)->exists()) {
			$isUnique = true;
		}

		// Register view.
		$view = $this->views()->create([
			'user_id' => $userId,
		]);

		// Update statistics.
		$this->updateViewStatistics(
			totalDelta: +1,
			uniqueDelta: $isUnique ? +1 : 0
		);

		return $view;
	}

	/**
	 * Total number of views.
	 * 
	 * @return Attribute
	 */
	public function totalViewsCount () : Attribute
	{
		return Attribute::get(fn () => $this->viewStatistics?->count_total ?? 0);
	}

	/**
	 * Total number of views.
	 * 
	 * @return Attribute
	 */
	public function uniqueViewsCount () : Attribute
	{
		return Attribute::get(fn () => $this->viewStatistics?->count_unique ?? 0);
	}

	/**
	 * Check if the viewable has any views.
	 * 
	 * @return boolean
	 */
	public function hasViews () : bool
	{
		return ($this->viewStatistics?->count_total ?? 0) > 0;
	}

	/**
	 * Update like statistics.
	 * 
	 * @param  integer  $delta
	 * 
	 * @return void
	 */
	public function updateViewStatistics (int $totalDelta, int $uniqueDelta) : void
	{
		$statistic = ViewStatistic::firstOrCreate([
			'viewable_type' => get_class($this),
			'viewable_id' => $this->id,
		]);

		$statistic->update([
			'count_total'  => max(0, $statistic->count_total + $totalDelta),
			'count_unique' => max(0, $statistic->count_unique + $uniqueDelta),
		]);
	}

}
