<?php

namespace JLGR\BugReports\Filament;

use Filament\Support\Enums\FontWeight;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use JLGR\BugReports\Enums\Status;
use JLGR\BugReports\Models\BugReport;


class AdminTableBuilder
{

	/**
	 * Build table renderer.
	 * 
	 * @param  Table  $table
	 * 
	 * @return Table
	 */
	public static function get (Table $table) : Table
	{
		return $table
			->query(static::getQuery())
			->columns(static::getColumns())
			->filters(static::getFilters())
			->actions(static::getActions())
			->defaultSort('created_at', 'desc');
	}

	/**
	 * Build query for the tables data source.
	 * 
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	protected static function getQuery () : Builder
	{
		return BugReport::query()
			->with([
				'user'
			]);
	}

	/**
	 * Define columns for the table grid.
	 * 
	 * @return array
	 */
	protected static function getColumns () : array
	{
		return [
			Columns\TextColumn::make('created_at')
				->label(__('Date'))
				->date('d-m-Y')
				->sortable(),
			Columns\TextColumn::make('name')
				->label(__('Subject'))
				->sortable()
				->searchable()
				->weight(FontWeight::Bold),
			Columns\TextColumn::make('description')
				->label(__('Description'))
				->words(7)
				->searchable(),
			Columns\TextColumn::make('user.name')
				->label(__('Requested by'))
				->url(fn (BugReport $record) : string 
					=> route('members.edit', [ 'member' => $record->user_id ]))
				->sortable()
				->searchable(),
			Columns\ViewColumn::make('status')
				->label(__('Status'))
				->view('user-reports::partials.status')
				->sortable()
				->state(fn (BugReport $record) : BugReport => $record),
		];
	}

	/**
	 * Define filters for this table.
	 * 
	 * @return array
	 */
	protected static function getFilters () : array
	{
		return [
			SelectFilter::make('status')
				->label(__('Status'))
				->options([
					Status::New->value => Status::New->getLabel(),
					Status::Pending->value => Status::Pending->getLabel(),
					Status::Confirmed->value => Status::Confirmed->getLabel(),
					Status::OnHold->value => Status::OnHold->getLabel(),
					Status::Scheduled->value => Status::Scheduled->getLabel(),
					Status::Resolved->value => Status::Resolved->getLabel(),
					Status::Rejected->value => Status::Rejected->getLabel(),
				])
		];
	}

	/**
	 * Define filters for this table.
	 * 
	 * @return array
	 */
	protected static function getActions () : array
	{
		return [
			// Action::make('edit')
			// 	->icon('heroicon-o-pencil')
			// 	->iconButton()
			// 	->url(
			// 		fn (Entity $record) : string
			// 			=> route('entities.edit', [ 'entityCategory' => $record->category->key, 'entity' => $record->id ])
			// 	),
		];
	}

}
