<?php

namespace JLGR\SystemMessages\Filament;

use Filament\Support\Enums\FontWeight;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use JLGR\SystemMessages\Enums\Status;
use JLGR\SystemMessages\Models\SystemMessage;

class SystemMessagesTableBuilder
{

	/**
	 * Build table renderer.
	 * 
	 * @param  Table  $table
	 * 
	 * @return Table
	 */
	public static function get (Table $table) : Table
	{
		return $table
			->query(static::getQuery())
			->columns(static::getColumns())
			->filters(static::getFilters())
			->actions(static::getActions())
			->defaultSort('title', 'asc');
	}

	/**
	 * Build query for the tables data source.
	 * 
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	protected static function getQuery () : Builder
	{
		return SystemMessage::query();
	}

	/**
	 * Define columns for the table grid.
	 * 
	 * @return array
	 */
	protected static function getColumns () : array
	{
		return [
			Columns\TextColumn::make('title')
				->label(__('Title'))
				->sortable()
				->searchable()
				->weight(FontWeight::Bold),
			Columns\TextColumn::make('icon')
				->label(__('Icon')),
			Columns\TextColumn::make('visible_from')
				->label(__('Visible from'))
				->dateTime('d-m-Y H:i')
				->sortable(),
			Columns\TextColumn::make('visible_until')
				->label(__('Visible until'))
				->dateTime('d-m-Y H:i')
				->sortable(),
			Columns\ViewColumn::make('status')
				->label(__('Status'))
				->view('system-messages::partials.status')
				->sortable()
				->state(fn (SystemMessage $record) : SystemMessage => $record),
		];
	}

	/**
	 * Define filters for this table.
	 * 
	 * @return array
	 */
	protected static function getFilters () : array
	{
		return [
			SelectFilter::make('status')
				->label(__('Status'))
				->options([
					Status::Draft->value => Status::Draft->getLabel(),
					Status::Published->value => Status::Published->getLabel(),
					Status::Unpublished->value => Status::Unpublished->getLabel(),
				])
		];
	}

	/**
	 * Define filters for this table.
	 * 
	 * @return array
	 */
	protected static function getActions () : array
	{
		return [
			Action::make('edit')
				->icon('heroicon-o-pencil')
				->iconButton()
				->url(
					fn (SystemMessage $record) : string
						=> route('system-messages.edit', [ 'systemMessage' => $record->id ])
				),
		];
	}

}
