<?php

namespace JLGR\Reviews\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\{ BelongsTo, HasMany, MorphTo };
use JLGR\Kernel\Models\BaseModel;
use JLGR\Reviews\Enums\ReviewStatus;

class Review extends BaseModel
{

	/**
	 * The table associated with the model.
	 * 
	 * @var string
	 */
	protected $table = 'reviews';

	/**
	 * The attributes that are protected from mass assignment.
	 *
	 * @var array<int, string>
	 */
	protected $guarded = [];

	/**
	 * Get the attributes that should be cast.
	 *
	 * @return array<string, string>
	 */
	protected function casts () : array
	{
		return [
			'status' => ReviewStatus::class,
		];
	}

    /**
     * Scope to retrieve only approved reviews.
     */
    public function scopeApproved (Builder $query)
    {
        return $query->where('status', ReviewStatus::Approved);
    }

	/**
	 * Get the parent reviewable model (museum, etc).
	 */
	public function reviewable () : MorphTo
	{
		return $this->morphTo();
	}

	/**
	 * Get the user that owns the like.
	 */
	public function user () : BelongsTo
	{
		return $this->belongsTo(config('jlgr.likes.user_model'));
	}

	/**
	 * Get all reports associated with this comment.
	 */
	public function reports () : HasMany
	{
		return $this->hasMany(ReviewReport::class);
	}

	/**
	 * Check if the comment is reported by the given user.
	 *
	 * @param  object  $user
	 * 
	 * @return bool
	 */
	public function isReportedBy (object $user) : bool
	{
		return $this->reports()->where('user_id', $user->id)->exists();
	}

	/**
	 * Count how many times this comment has been reported.
	 *
	 * @return int
	 */
	public function reportCount () : int
	{
		return $this->reports()->count();
	}

    /**
     * Get the ratings associated with this review.
     */
    public function ratings () : HasMany
    {
        return $this->hasMany(ReviewRating::class);
    }

    /**
     * Check if the review has been approved.
     */
    public function isApproved () : bool
    {
        return $this->status === ReviewStatus::Approved;
    }

	/**
	 * Recalculate avarage rating for this review.
	 * 
	 * @return void
	 */
	public function updateAverageRating () : void
	{
		$avg = $this->ratings()->avg('score');

		$this->update([
			'rating_average' => round($avg, 2)
		]);
	}

}
