<?php

namespace JLGR\Reviews\Livewire;

use Illuminate\Database\Eloquent\Collection;
use Livewire\Component;
use Illuminate\Database\Eloquent\Model;
use JLGR\Reviews\Contracts\ReportsReview;
use JLGR\Reviews\Enums\ReviewStatus;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;

class ReviewsList extends Component
{
	
	/**
	 * The reviewable model.
	 */
	public Model $reviewable;

	/**
	 * Whether to show unapproved reviews.
	 */
	public bool $includeUnapproved = false;
	
	/**
	 * The ID of the review currently being reported.
	 * When set, the reply form will be shown under this review.
	 *
	 * @var int|null
	 */
	public ?int $reportingReview = null;

    /**
     * Maximum number of reviews to display. If null, all reviews
	 * will be shown.
     *
     * @var int|null
     */
    public ?int $limit = null;

	/**
	 * Component properties.
	 * 
	 * @var mixed
	 */
	public string $template = 'default';

	/**
	 * Mount the component.
	 */
	public function mount (Model $reviewable) : void
	{
		$this->reviewable = $reviewable;
	}

	/**
	 * Load all top-level reviews.
	 */
	#[ On('review-created') ]
	public function refresh () : void
	{
		$this->dispatch('$refresh');
	}

	/**
	 * Computed property:
	 * Reviews, optionally filtered on approval.
	 * 
	 * @return Collection<int, Review>
	 */
	#[ Computed() ]
	public function reviews () : Collection
	{
		// Start building the query for top-level reviews
		$query = $this->reviewable
			->reviews()
			->with([
				'user.media',
			]);

		// Filter out unapproved reviews if the flag is disabled
		if (! $this->includeUnapproved) {
			$query->whereNot('status', ReviewStatus::Pending);
		}

		// Limit the number of reviews if specified
		if (! is_null($this->limit)) {
			$query->limit($this->limit);
		}

		return $query->latest()->get();
	}

	/**
	 * Report review as inappropriate.
	 *
	 * @param  int  $reviewId
	 * 
	 * @return void
	 */
	public function report (int $reviewId, ReportsReview $reviewReporter): void
	{
		$reviewReporter->handle($reviewId);

		$this->reportingReview = null;
	}

	/**
	 * Toggle visibility of the report review form.
	 *
	 * @param  int  $reviewId
	 * 
	 * @return void
	 */
	public function toggleReport (int $reviewId) : void
	{
		$this->reportingReview = $this->reportingReview === $reviewId
			? null
			: $reviewId;
	}

	/**
	 * Render the component.
	 */
	public function render ()
	{
		return view("reviews::livewire.reviews-list.{$this->template}", [
			'reviews' => $this->reviews,
		]);
	}

}
