<?php

namespace JLGR\Reviews\Livewire;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Carbon;
use JLGR\Reviews\Models\Review;
use Livewire\Component;

class LatestReview extends Component
{

	/**
	 * Random review.
	 * 
	 * @var Review|null
	 */
	public ?Review $review;

	/**
	 * Component properties.
	 * 
	 * @var mixed
	 */
	public string $template = 'default';

	/**
	 * Mount the component.
	 */
	public function mount () : void
	{
		$this->review = $this->getRandomReview();
	}

	/**
	 * Retrieve random - or latest - review from the database.
	 * 
	 * @return Review|null
	 */
	protected function getRandomReview () : ?Review
	{
		try
		{
			return Review::query()
				->with([
					'user',
					'reviewable'
				])
				->where('created_at', '>=', Carbon::now()->subWeeks(2))
				->inRandomOrder()
				->firstOrFail();
		}
		catch (ModelNotFoundException $exception)
		{
			return Review::query()
				->with([
					'user',
					'reviewable'
				])
				->latest()
				->first();
		}
	}

	/**
	 * Render the component.
	 */
	public function render ()
	{
		return view("reviews::livewire.latest-review.{$this->template}");
	}

}
