<?php

namespace JLGR\Reviews\Concerns;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\{ MorphMany, MorphOne };
use Illuminate\Support\Facades\Auth;
use JLGR\Reviews\Models\Review;
use JLGR\Reviews\Models\ReviewIndicator;
use JLGR\Reviews\Models\ReviewStatistic;

trait CanBeReviewed
{

	/**
	 * Get all reviews for the model.
	 */
	public function reviews () : MorphMany
	{
		return $this->morphMany(Review::class, 'reviewable')->latest();
	}
	
	/**
	 * Get the review indicators for this reviewable type.
	 */
	public function reviewIndicators () : Collection
	{
		return ReviewIndicator::query()
			->where('reviewable_type', static::class)
			->get();
	}

	/**
	 * Get indicators as [name => label] array.
	 */
	public function reviewIndicatorOptions (): array
	{
		return $this->reviewIndicators()
			->pluck('label', 'name')
			->toArray();
	}

	/**
	 * Get statistics for this model.
	 */
	public function reviewStatistics () : MorphOne
	{
		return $this->morphOne(ReviewStatistic::class, 'reviewable');
	}

	/**
	 * Get the number of reviews for this reviewable.
	 * 
	 * @return integer
	 */
	public function reviewsCount () : int
	{
		return intval($this->reviewStatistics?->count ?? 0);
	}

	/**
	 * Get the average rating for this reviewable.
	 * 
	 * @param  boolean  $formatted
	 * 
	 * @return float|string|null
	 */
	public function averageReviewRating (bool $formatted = false) : float|string|null
	{
		$rating = $this->reviewStatistics?->average_rating ?? 0;

		return ($formatted == true)
			? number_format($rating, 2, ',', '.')
			: $rating;
	}

	/**
	 * Check if the reviewable is reviewed by the authenticated user.
	 * 
	 * @return boolean
	 */
	public function isReviewed () : bool
	{
		$userId = Auth::id();

		return $userId
			? $this->reviews()->where('user_id', $userId)->exists()
			: false;
	}

	/**
	 * Check if the reviewable has any reviews.
	 * 
	 * @return boolean
	 */
	public function hasReviews () : bool
	{
		return $this->reviewsCount() > 0;
	}

	/**
	 * Update like statistics.
	 * 
	 * @param  integer  $delta
	 * 
	 * @return void
	 */
	public function updateReviewStatistics (int $delta) : void
	{
		$statistic = ReviewStatistic::firstOrCreate([
			'reviewable_type' => get_class($this),
			'reviewable_id' => $this->id,
		]);

		// Update count.
		$newCount = max(0, $statistic->count + $delta);

		// Calculate average rating (only approved reviews).
		$averageRating = $this->reviews()
			->approved()
			->avg('average_rating');

		$statistic->update([
			'count' => $newCount,
			'average_rating' => round($averageRating, 2),
		]);
	}

}
