<?php

namespace JLGR\Reviews\Actions;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use JLGR\Reviews\Contracts\CreatesReview;
use JLGR\Reviews\Enums\ReviewStatus;
use JLGR\Reviews\Models\ReviewIndicator;
use JLGR\Reviews\Models\ReviewRating;

class CreateReview implements CreatesReview
{

	/**
	 * Create new review.
	 *
	 * @param  mixed  $reviewable
	 * @param  array  $data
	 * 
	 * @return void
	 */
	public function handle (mixed $reviewable, array $data) : void
	{
		// Wrap all data mutations in a single database transaction. In
		// case of an error, everything will be rolled back. 
		DB::transaction(function () use ($reviewable, $data) {
			$ratings = $data['ratings'] ?? [];

			// Get all relevant indicators (1 query).
			$indicators = ReviewIndicator::where('reviewable_type', get_class($reviewable))
				->whereIn('key', array_keys($ratings))
				->get()
				->keyBy('key');

			// Calculate average score.
			$total = 0;
			$count = 0;

			foreach ($ratings as $indicatorKey => $value) {
				if (! isset($indicators[$indicatorKey])) {
					continue;
				}
				$total += (int) $value;
				$count++;
			}

			// Calculate...
			$average = $count > 0 ? round($total / $count, 2) : null;

			// Determine status.
			$status = config('jlgr.reviews.auto_approve') 
				? ReviewStatus::Approved 
				: ReviewStatus::Pending;

			// Create review.
			$review = $reviewable->reviews()->create([
				'user_id' => Auth::id(),
				'text' => $data['text'],
				'status' => $status,
				'average_rating' => $average,
			]);

			$insertData = [];
			$timestamp = Carbon::now();

			// Process ratings.
			foreach ($ratings as $indicatorKey => $score)
			{
				if (! isset($indicators[$indicatorKey])) {
					continue;
				}

				$insertData[] = [
					'review_id' => $review->id,
					'review_indicator_id' => $indicators[$indicatorKey]->id,
					'score' => (int) $score,
					'created_at' => $timestamp,
					'updated_at' => $timestamp,
				];
			}

			// Insert ratings.
			if (! empty($insertData)) {
				ReviewRating::insert($insertData);
			}

			// Update statistics.
			$reviewable->updateReviewStatistics(+1);
		});
	}

}
