<?php

namespace JLGR\ReleaseNotes\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use JLGR\Kernel\Models\BaseModel;
use JLGR\ReleaseNotes\Models\ReleaseNote;

class Release extends BaseModel
{

	/**
	 * The table associated with the model.
	 * 
	 * @var string
	 */
	protected $table = 'releases';

	/**
	 * The attributes that are protected from mass assignment. An empty
	 * array means all fields are mass assignable.
	 * 
	 * @var array
	 */
	protected $guarded = [];

	/**
     * Query scope for version ordering.
     *
     * @param Builder $query
     * @param string $order
     *
     * @return Builder
     */
    public function scopeOrderByVersion ($query, $order = 'desc')
	{
		return $query
			->orderBy('major', $order)
			->orderBy('minor', $order)
			->orderBy('patch', $order);
	}

	/**
	 * All release notes that belong to this release.
	 */
	public function notes ()
	{
		return $this->hasMany(ReleaseNote::class);
	}

	/**
	 * Group notes by type.
	 */
	public function getGroupedNotesAttribute ()
	{
		return $this->notes
			->mapToGroups(fn ($note)
				=> [ $note->type => $note ]);
	}

	/**
	 * Format version number.
	 * 
	 * @return string
	 */
	public function getVersionAttribute ()
	{
		// Version segments.
		$segments = [
			$this->major,
			$this->minor,
			$this->patch,
		];

		// Remove null values, but not the 0 version value.
		$segments = array_filter($segments, function($value) {
			return ($value !== null and $value !== false and $value !== ''); 
		});

		// Return version string.
		return implode('.', $segments);
	}

	/**
	 * Return the properly formatted departure time.
	 * 
	 * @return string
	 */
	public function getReleaseDateForHumansAttribute ()
	{
		return formatDate($this->created_at);
	}

}
