<?php

namespace JLGR\MemberPoints\Concerns;

use Illuminate\Database\Eloquent\Relations\HasMany;
use JLGR\MemberPoints\Models\MemberPoint;
use JLGR\MemberPoints\Enums\MemberPoints;
use JLGR\MemberPoints\Enums\Status;

trait CollectsPoints
{
	
	/**
	 * Grant points to the user.
	 */
	public function grantPoints (
		MemberPoints $type,
		?object $pointable = null,
		bool $unique = false,
		?string $notes = null
	) : void
	{
		if ($unique && $this->hasGrantedPoints($type, $pointable)) {
			return;
		}

		MemberPoint::create([
			'user_id' => $this->id,
			'type' => $type,
			'points' => $type->points(),
			'is_reversal' => false,
			'status' => Status::Published,
			'notes' => $notes,
			'pointable_id' => $pointable?->id,
			'pointable_type' => $pointable ? get_class($pointable) : null,
		]);
	}

	/**
	 * Revoke previously granted points by creating a reversal entry.
	 * 
	 * @param  MemberPoints  $type
	 * @param  ?object  $pointable
	 * @param  ?string  $notes
	 */
	public function revokePoints (
		MemberPoints $type,
		?object $pointable = null,
		?string $notes = null
	) : void
	{
		$original = $this->memberPoints()
			->where('type', $type)
			->where('is_reversal', false)
			->when($pointable, fn ($query)
				=> $query
					->where('pointable_id', $pointable->id)
				  	->where('pointable_type', get_class($pointable)))
			->latest()
			->first();

		MemberPoint::create([
			'user_id' => $this->id,
			'type' => $type,
			'points' => -1 * $type->points(),
			'is_reversal' => true,
			'reversal_of_id' => $original?->id,
			'status' => Status::Published,
			'notes' => $notes,
			'pointable_id' => $pointable?->id,
			'pointable_type' => $pointable ? get_class($pointable) : null,
		]);
	}

	/**
	 * Check if points have already been granted for a specific type and object.
	 * 
	 * @param  MemberPoints  $type
	 * @param  ?object  $pointable
	 * 
	 * @return boolean
	 */
	public function hasGrantedPoints (MemberPoints $type, ?object $pointable = null) : bool
	{
		return $this->memberPoints()
			->where('type', $type)
			->where('is_reversal', false)
			->where('status', Status::Published)
			->when($pointable, fn($query)
				=> $query
					->where('pointable_id', $pointable->id)
				  	->where('pointable_type', get_class($pointable))
			)
			->exists();
	}

	/**
	 * The member points that belong to this user.
	 */
	public function memberPoints () : HasMany
	{
		return $this->hasMany(MemberPoint::class);
	}

}
