<?php

namespace JLGR\MemberPoints;

use AviatorsEcho\Models\User;
use JLGR\MemberPoints\Enums\MemberPoints as MemberPointsEnum;
use JLGR\MemberPoints\Models\MemberPoint;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Carbon;
use JLGR\MemberPoints\Enums\Status;

class MemberPointService
{

	/**
	 * The user who receives the points.
	 * 
	 * @var User
	 */
	protected User $user;

	/**
	 * The type of granted points.
	 * 
	 * @var MemberPointsEnum
	 */
	protected MemberPointsEnum $type;

	/**
	 * Entity related to the granted points.
	 * 
	 * @var object
	 */
	protected ?object $pointable = null;

	/**
	 * Optional notes.
	 * 
	 * @var string
	 */
	protected ?string $notes = null;

	/**
	 * Flag indicating the points should be granted just once.
	 * 
	 * @var boolean
	 */
	protected bool $unique = false;

	/**
	 * Flag indicating the points should be granted just once per day.
	 * 
	 * @var boolean
	 */
	protected bool $uniqueForToday = false;

	/**
	 * Create static instance of the service.
	 * 
	 * @return self
	 */
	public static function make () : self
	{
		return new self();
	}

	/**
	 * Grant points
	 * 
	 * @param  User  $user,
	 * @param  MemberPointsEnum  $type,
	 * @param  ?object  $pointable 
	 * @param  bool  $unique 
	 * @param  ?string  $notes 
	 * 
	 * @return void
	 */
	public static function grantPoints (
		User $user,
		MemberPointsEnum $type,
		?object $pointable = null,
		bool $unique = false,
		?string $notes = null
	): void {
		self::make()
			->forUser($user)
			->forType($type)
			->withPointable($pointable)
			->withNotes($notes)
			->ensureUnique($unique)
			->grant();
	}

	/**
	 * Set user on the instance.
	 * 
	 * @param  User|integer  $user
	 * 
	 * @return self
	 */
	public function forUser (User|int $user) : self
	{
		if (is_int($user)) {
			$user = User::findOrFail($user);
		}

		$this->user = $user;
		return $this;
	}

	/**
	 * Set type on the instance.
	 * 
	 * @param  MemberPointsEnum  $type
	 * 
	 * @return self
	 */
	public function forType (MemberPointsEnum $type) : self
	{
		$this->type = $type;
		return $this;
	}

	/**
	 * Set the pointable object on the instance.
	 * 
	 * @param  object  $pointable
	 * 
	 * @return self
	 */
	public function withPointable (?object $pointable) : self
	{
		$this->pointable = $pointable;
		return $this;
	}

	/**
	 * Set notes on the instance.
	 * 
	 * @param  string  $notes
	 * 
	 * @return self
	 */
	public function withNotes (?string $notes) : self
	{
		$this->notes = $notes;
		return $this;
	}

	/**
	 * Ensure the granting is unique.
	 * 
	 * @param  boolean  $unique
	 * 
	 * @return self
	 */
	public function ensureUnique (bool $unique = true) : self
	{
		$this->unique = $unique;
		return $this;
	}

	/**
	 * Ensure the granting is unique for the date of today.
	 * 
	 * @param  boolean  $uniqueForToday
	 * 
	 * @return self
	 */
	public function ensureUniqueForToday (bool $uniqueForToday = true) : self
	{
		$this->uniqueForToday = $uniqueForToday;
		return $this;
	}

	/**
	 * Grant points (create member point record).
	 * 
	 * @return MemberPoint
	 */
	public function grant () : ?MemberPoint
	{
		if ($this->unique && $this->alreadyGranted()) {
			return null;
		}

		if ($this->uniqueForToday && $this->alreadyGrantedForToday()) {
			return null;
		}

		return MemberPoint::create([
			'user_id' => $this->user->id,
			'type' => $this->type,
			'points' => $this->type->points(),
			'pointable_id' => $this->pointable?->id,
			'pointable_type' => $this->pointable ? get_class($this->pointable) : null,
			'is_reversal' => false,
			'reversal_of_id' => null,
			'status' => Status::Published,
			'notes' => $this->notes,
		]);
	}

	/**
	 * Revoke points.
	 * 
	 * @return void
	 */
	public function revoke () : void
	{
		MemberPoint::query()
			->where('user_id', $this->user->id)
			->where('type', $this->type)
			->where('pointable_id', $this->pointable?->id)
			->where('pointable_type', $this->pointable ? get_class($this->pointable) : null)
			->get()
			->each(fn (MemberPoint $memberPoint)
				=> $memberPoint->update([
					   'status' => Status::Unpublished
				   ])
			);
	}

	/**
	 * Check if the points already have been granted.
	 * 
	 * @return boolean
	 */
	protected function alreadyGranted () : bool
	{
		return $this->alreadyGrantedQuery()->exists();
	}

	/**
	 * Check if the points already have been granted for today.
	 * 
	 * @return boolean
	 */
	protected function alreadyGrantedForToday () : bool
	{
		return $this
			->alreadyGrantedQuery()
			->whereDate('created_at', Carbon::today())
			->exists();
	}

	/**
	 * Build query for checking if the granting already has been done.
	 * 
	 * @return HasMany
	 */
	protected function alreadyGrantedQuery () : HasMany
	{
		return $this->user
			->memberPoints()
			->where('type', $this->type)
			->where('is_reversal', false)
			->where('status', Status::Published)
			->when($this->pointable, fn ($query) 
				=> $query
					->where('pointable_id', $this->pointable->id)
				  	->where('pointable_type', get_class($this->pointable)));
	}

}
