<?php

namespace JLGR\MediaManagement\Livewire;

use Filament\Actions\Contracts\HasActions;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Schemas\Schema;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Illuminate\Database\Eloquent\Model;
use InvalidArgumentException;
use JLGR\MediaManagement\DTOs\UploadContext;
use JLGR\MediaManagement\Services\MediaManagerService;
use JLGR\MediaManagement\Support\MediaTypeMapper;
use Livewire\Component;

class MediaUploader extends Component implements HasForms, HasActions
{
	use InteractsWithActions;
    use InteractsWithForms;

	/**
	 * The model to attach uploaded media to.
	 */
	public Model $model;

	/**
	 * Upload context.
	 */
	public array $context;

	/**
	 * Form state.
	 */
	public array $data = [];

	/**
	 * Resolve and initialize configuration for this media type.
	 * 
	 * @param  array  $context
	 */
	public function mount (array $context) : void
	{
		if (!isset($context) || !is_array($context)) {
			throw new InvalidArgumentException('The $context property is required and must be an array.');
		}

		$this->context = $context;
	}

	/**
	 * Build the Filament form schema.
	 */
	public function form (Schema $schema) : Schema
	{
		$config = MediaTypeMapper::resolve($this->context['type']);

		$formBuilderClass = $config['form'];

		return $formBuilderClass::make($schema, $this->context);
	}

	/**
	 * Process the uploaded file using the correct handler.
	 *
	 * @param  MediaManagerService  $mediaManager
	 */
	public function processForm (MediaManagerService $mediaManager) : void
	{
		$this->form->validate();
		
		$mediaManager->upload(
			model: $this->model,
			data: $this->data,
			context: $this->getUploadContext()
		);

		$this->dispatch('media-uploaded', type: $this->context['type']);
	}

	/**
	 * Lazily convert the public array into a type-safe DTO.
	 *
	 * @throws InvalidArgumentException
	 */
	protected function getUploadContext () : UploadContext
	{
		// Validate required keys.
		$requiredKeys = [ 'type', 'collection', 'multiple', 'acceptedMimes', 'maxSize' ];
		$missing = array_filter($requiredKeys, fn($key) => !array_key_exists($key, $this->context));

		if ($missing) {
			throw new InvalidArgumentException('Upload context array is missing required keys: ' . implode(', ', $missing));
		}

		// convert to DTO
		return UploadContext::fromArray($this->context);
	}

	/**
	 * Render the Livewire view.
	 */
	public function render ()
	{
		return view('media-management::livewire.media-uploader.default');
	}

}
