<?php

namespace JLGR\Likes\Livewire;

use Illuminate\Database\Eloquent\Collection;
use Livewire\Component;
use Illuminate\Database\Eloquent\Model;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;

class Likers extends Component
{
	
	/**
	 * The likeable model.
	 */
	public Model $likeable;

	/**
	 * Maximum number of likers to display.
	 *
	 * @var int
	 */
	public int $limit = 3;

	/**
	 * Total numer of likes.
	 *
	 * @var boolean
	 */
	public bool $showLikesCount = false;

	/**
	 * Total number of likes.
	 *
	 * @var array
	 */
	public array $randomLikers = [];

	/**
	 * Component properties.
	 * 
	 * @var mixed
	 */
	public string $template = 'default';

	/**
	 * Mount the component.
	 * 
	 * @param  Model  $likeable
	 * @param  integer  $limit
	 */
	public function mount (Model $likeable, int $limit = 3, bool $showLikesCount = false) : void
	{
		$this->likeable = $likeable;
		$this->limit = $limit;
		$this->showLikesCount = $showLikesCount;
	}

	/**
	 * Re-render component.
	 */
	#[ On('like-toggled') ]
	public function refreshLikers () : void
	{
		$this->dispatch('$refresh');
	}

	/**
	 * Computed property: likers.
	 * 
	 * @return Collection<int, Comment>
	 */
	#[ Computed() ]
	public function likes () : Collection
	{
		// Start building the query n number of likers.
		$query = $this->likeable
			->likes()
			->with([
				'user.media'
			])
			->inRandomOrder();

		// Limit the number of likers if specified
		if (! is_null($this->limit)) {
			$query->limit($this->limit);
		}

		return $query->latest()->get();
	}

	/**
	 * Computed property: likers.
	 * 
	 * @return integer
	 */
	#[ Computed() ]
	public function likesCount () : int
	{
		return ($this->showLikesCount == true)
			? $this->likeable->getLikesCount()
			: 0;
	}

	/**
	 * Render the component.
	 */
	public function render ()
	{
		return view("likes::livewire.likers.{$this->template}", [
			'likes' => $this->likes,
			'likesCount' => $this->likesCount,
		]);
	}

}
