<?php

namespace JLGR\Likes\Concerns;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Facades\Auth;
use JLGR\Likes\Models\Like;
use JLGR\Likes\Models\LikeStatistic;

trait CanBeLiked
{

	/**
	 * Get all likes for the model.
	 */
	public function likes () : MorphMany
	{
		return $this->morphMany(Like::class, 'likeable')->latest();
	}

	/**
	 * Get statistics for this model.
	 */
	public function likeStatistics () : MorphOne
	{
		return $this->morphOne(LikeStatistic::class, 'likeable');
	}

	/**
     * Add a like to this model.
     *
     * @param  integer  $userId
     *
     * @return Like|null
     */
    public function addLike (?int $userId = null) : ?Like
	{
		$userId = $userId ?? Auth::id();
		
		// Silent fail: prevent user from liking own model.
		if (
			method_exists($this, 'user') &&
			($owner = $this->user()?->getResults()) &&
			$owner->id === $userId
		) {
			return null;
		}
		
		$like = $this->likes()->create([
			'user_id' => $userId,
		]);

		$this->updateLikeStatistics(+1);

		return $like;
	}

	/**
	 * Remove a like from this model.
	 * 
	 * @param  integer  $userId
	 * 
	 * @return void
	 */
	public function removeLike (?int $userId = null) : void
	{
		$like = $this->likes()->where('user_id', $userId ?? Auth::id())->first();

		if ($like) {
			$like->delete();
			$this->updateLikeStatistics(-1);
		}
	}

	/**
	 * Toggle like for this model.
	 * 
	 * @param  integer  $userId
	 * 
	 * @return void
	 */
	public function toggleLike (?int $userId = null) : void
	{
		$userId = $userId ?? Auth::id();

		$existing = $this->likes()->where('user_id', $userId)->first();

		if ($existing) {
			$this->removeLike($userId);
		} else {
			$this->addLike($userId);
		}
	}

	/**
	 * Get the number of likes for this likeable.
	 * 
	 * @return integer
	 */
	public function getLikesCount () : int
	{
		return intval($this->likeStatistics?->count ?? 0);
	}
	
	/**
	 * Accessor for returning the number of likes for this model.
	 */
	public function likesCount () : Attribute
	{
		return Attribute::get(
			fn () => $this->getLikesCount()
		);
	}

	/**
	 * Check if the likeable is liked by the authenticated user.
	 * 
	 * @return boolean
	 */
	public function isLiked () : bool
	{
		if (Auth::check()) {
			return $this->likes()->where('user_id', Auth::id())->exists();
		}

		return false;
	}

	/**
	 * Check if the likeable has any likes.
	 * 
	 * @return boolean
	 */
	public function hasLikes () : bool
	{
        return ($this->likeStatistics?->count ?? 0) > 0;
	}

	/**
	 * Update like statistics.
	 * 
	 * @param  integer  $delta
	 * 
	 * @return void
	 */
	public function updateLikeStatistics (int $delta) : void
	{
		$statistic = LikeStatistic::firstOrCreate([
			'likeable_type' => get_class($this),
			'likeable_id' => $this->id,
		]);

		$newCount = max(0, $statistic->count + $delta);
		$statistic->update(['count' => $newCount]);
	}

}
