<?php
namespace JLGR\Kernel\Repositories;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use JLGR\Kernel\Enums\CacheDuration;
use RuntimeException;

abstract class BaseRepository
{

	/**
	 * Indicates whether caching is enabled.
	 * 
	 * @var boolean
	 */
	protected bool $useCaching = true;

	/**
	 * The cache tag for grouping cached data.
	 *
	 * @var string
	 */
	protected string $cacheTag;

	/**
	 * Retrieve from cache or cache a query if it doesn't exist.
	 *
	 * @param  string  $cacheKey
	 * @param  \Closure  $callback
	 * @param  int|null  $userId
	 * @param  int  $hours
	 * @param  int  $minutes
	 * 
	 * @return mixed
	 */
	protected function retrieveFromCache (
		string $cacheKey,
		\Closure $callback,
		array $tags = [],
		CacheDuration $duration = CacheDuration::OneDay,
	) : mixed
	{
		$tags = $tags ?: [ $this->cacheTag ];
		$expiry = Carbon::now()->addMinutes($duration->value);
		
		return $this->useCaching

			// Caching enabled: return from cache or cache new data and then return.
			? Cache::tags($tags)->remember($cacheKey, $expiry, $callback)

			// Execute closure and return the results without caching.
			: $callback();
	}

	/**
	 * Clears all cache items for the specified tag.
	 * 
	 * @throws RuntimeException
	 * 
	 * @return void
	 */
	public function clearCache () : void
	{
		if ( ! isset($this->cacheTag)) {
			throw new RuntimeException('Cache tag is not defined.');
		}

		Cache::tags([$this->cacheTag])->flush();
	}

	/**
	 * Generate a cache key based on parameters.
	 * 
	 * @param  array  $params
	 * 
	 * @return string
	 */
	protected function generateParamsHash (array $params) : string
	{
		return md5(json_encode($params));
	}

	/**
	 * Generates a unique cache key based on filters, pagination, and sorting parameters.
	 *
	 * @param  array  $filters Filters applied to the query.
	 * @param  integer  $page The page number.
	 * @param  integer  $perPage The number of items per page.
	 * @param  string  $sort The column to sort by.
	 * @param  string  $direction The sort direction (asc or desc).
	 * 
	 * @return string  The generated cache key.
	 */
	protected function generateCacheKey (
		array $filters,
		int $page,
		int $perPage,
		string $sort,
		string $direction
	) : string
	{
		// Convert filters to query string format.
		$filterString = http_build_query($filters);
		
		// Concat parameters to key string.
		return "page:{$page}:perPage:{$perPage}:sort:{$sort}:direction:{$direction}:filters:{$filterString}";
	}

}
