<?php

namespace JLGR\Kernel\Queries\Eloquent;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\Cursor;
use Illuminate\Support\Facades\Auth;
use InvalidArgumentException;

class BaseQuery
{

	/**
	 * Flag indicating relationships should be eager loaded by default.
	 * 
	 * @var boolean
	 */
	protected static $withRelations = true;

	/**
	 * Id of the user to get the number for.
	 * 
	 * @var integer
	 */
	protected static $userId = null;

	/**
	 * Create a new builder instance and check if the global user scope should
	 * be disabled (if applicable).
	 * 
	 * @param mixed $model Eloquent model. 
	 * 
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	protected static function newQuery ($model)
	{
		return $model::query();
	}

	/**
	 * Return Eloquent builder.
	 * 
	 * @param array $params Optional array with filter params.
	 * 
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public static function query (array $params = []) : Builder
	{
		return static::builder($params);
	}

	/**
	 * Return Eloquent collection.
	 * 
	 * @param  array  $params  Optional array with filter params.
	 * @param  integer  $limit  Optional limit number.
	 * 
	 * @return Collection
	 */
	public static function collection (array $params = [], $limit = null) : Collection
	{
		return ($limit)
			? static::builder($params)->limit($limit)->get()
			: static::builder($params)->get();
	}

	/**
	 * Return array with key-value pairs.
	 * 
	 * @param  array  $attributes  Array with the attributes to be returned.
	 * @param  array  $params  Optional array with filter params.
	 * 
	 * @return array
	 */
	public static function pluck (array $attributes, array $params = [])
	{
		$builder = static::builder($params)->get();
	
		switch (count($attributes)) {
			case 1:
				return $builder->pluck($attributes[0])->toArray();
			case 2:
				return $builder->pluck($attributes[0], $attributes[1])->toArray();
			default:
				throw new InvalidArgumentException('Pluck requires 1 or 2 attributes.');
		}
	}

	/**
	 * Return array.
	 * 
	 * @param  array  $params  Optional array with filter params.
	 * 
	 * @return array
	 */
	public static function array (array $params = []) : array
	{
		return static::builder($params)->get()->toArray();
	}

	/**
	 * Return single Eloquent model.
	 * 
	 * @param array $params Optional array with filter params.
	 * 
	 * @return Model|null
	 */
	public static function first (array $params = []) : Model|null
	{
		return static::builder($params)->first();
	}

	/**
	 * Return paginated collection with Eloquent models.
	 * 
	 * @param integer $perPage Optional integer with the number of items per page.
	 * @param array $params Optional array with filter params.
	 * 
	 * @return Illuminate\Database\Eloquent\Model
	 */
	public static function paginate ($perPage = 10, array $params = [])
	{
		return static::builder($params)->paginate($perPage);
	}

	/**
	 * Return paginated collection with Eloquent models.
	 * 
	 * @param  integer  $perPage Integer with the number of items per page.
	 * @param  string  $nextCursor
	 * @param  array  $params  Optional array with filter params.
	 * 
	 * @return Illuminate\Database\Eloquent\Model
	 */
	public static function cursorPaginate ($perPage, $nextCursor, array $params = [])
	{
		return static::builder($params)
			->cursorPaginate(
				$perPage,
				['*'],
				'cursor',
				Cursor::fromEncoded($nextCursor)
			);
	}

	/**
	 * Return fast paginated collection with Eloquent models.
	 * 
	 * @param integer $perPage Optional integer with the number of items per page.
	 * @param array $params Optional array with filter params.
	 * 
	 * @return Illuminate\Database\Eloquent\Model
	 */
	public static function fastPaginate ($perPage = 10, array $params = [])
	{
		return static::builder($params)->fastPaginate($perPage);
	}

	/**
	 * Return the nimber of models.
	 * 
	 * @param  array  $params  Optional array with filter params.
	 * @param  string  $column  Column identifier
	 * 
	 * @return Illuminate\Database\Eloquent\Model
	 */
	public static function count (array $params = [], $column = null)
	{
		return ($column)
			? static::builder($params)->count($column)
			: static::builder($params)->count();
	}

	/**
	 * Check if there are models present.
	 * 
	 * @param array $params Optional array with filter params.
	 * 
	 * @return boolean
	 */
	public static function exists (array $params = [])
	{
		return static::builder($params)->count() > 0;
	}

	/**
	 * Set the given user id on the base query class.
	 * 
	 * @param integer $userId
	 * 
	 * @return self
	 */
	public static function forUser ($userId = null)
	{
		static::$userId = $userId;

		return new static;
	}

	/**
	 * Set the authenticated user's id on the base query class.
	 * 
	 * @return self
	 */
	public static function forSelf ()
	{
		static::$userId = Auth::id();

		return new static;
	}

	/**
	 * Prevent relationships from being eager loaded.
	 * 
	 * @return self
	 */
	public static function withoutRelations ()
	{
		static::$withRelations = false;

		return new static;
	}

}
