<?php

namespace JLGR\Kernel\Queries\DB;

use Illuminate\Database\Query\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class BaseQuery
{
	
	/**
	 * Id of the user to get the data for.
	 *
	 * @var integer|null
	 */
	protected static ?int $userId = null;

	/**
	 * Create a new query builder instance for a given table.
	 *
	 * @param  string  $table  Table name to query.
	 * 
	 * @return Builder
	 */
	protected static function newQuery (string $table) : Builder
	{
		return DB::table($table);
	}

	/**
	 * Set the user ID for the query.
	 *
	 * @param  int  $userId
	 * 
	 * @return static
	 */
	public static function forUser (int $userId) : static
	{
		static::$userId = $userId;
		return new static();
	}

	/**
	 * Set the authenticated user's ID for the query.
	 * 
	 * @return static
	 */
	public static function forSelf () : static
	{
		static::$userId = Auth::user()->id;
		return new static();
	}

	/**
	 * Return a raw query builder.
	 *
	 * @param  array $params  Optional array with filter parameters.
	 * 
	 * @return Builder
	 */
	public static function query (array $params = []) : Builder
	{
		return static::builder($params);
	}

	/**
	 * Return a collection of records from the query.
	 *
	 * @param  array  $params  Optional array with filter parameters.
	 * 
	 * @return Collection
	 */
	public static function collection (array $params = []) : Collection
	{
		return static::builder($params)->get();
	}

	/**
	 * Return a single record from the query.
	 *
	 * @param  array  $params  Optional array with filter parameters.
	 * 
	 * @return object|null
	 */
	public static function first (array $params = []) : ?object
	{
		return static::builder($params)->first();
	}

	/**
	 * Pluck a single column from the query results.
	 *
	 * @param  string  $column  Column to pluck.
	 * @param  array  $params  Optional array with filter parameters.
	 * 
	 * @return array
	 */
	public static function pluck (string $column, array $params = []) : array
	{
		// Pluck the column and return only the values as a flat array
		return static::builder($params)->pluck($column)->all();
	}

	/**
	 * Get a single value from a column (like the first ID).
	 *
	 * @param  string  $column  Column to fetch the value from.
	 * @param  array  $params  Optional array with filter parameters.
	 * 
	 * @return mixed
	 */
	public static function value (string $column, array $params = []) : mixed
	{
		// Return the value of the column from the first row
		return static::builder($params)->value($column);
	}
	
	/**
	 * Pluck a key-value pair from the query results.
	 *
	 * @param  string  $keyColumn  Column to use as the key.
	 * @param  string  $valueColumn  Column to use as the value.
	 * @param  array  $params  Optional array with filter parameters.
	 * 
	 * @return array
	 */
	public static function pluckKeyValue (string $keyColumn, string $valueColumn, array $params = []) : array
	{
		return static::builder($params)->pluck($valueColumn, $keyColumn)->toArray();
	}

	/**
	 * Get the number of records.
	 * 
	 * @param  array  $params  Optional array with filter parameters.
	 * 
	 * @return integer
	 */
	public static function count (array $params = []) : int
	{
		// Return the count.
		return static::builder($params)->count();
	}

	/**
	 * Build the query. This method should be implemented in child classes.
	 *
	 * @param  array  $params  Optional array with filter parameters.
	 * @return Builder
	 */
	protected static function builder (array $params = []) : Builder
	{
		throw new \BadMethodCallException("The 'builder' method must be implemented in the child class.");
	}

	/**
	 * Reset the static user ID (useful for scoped queries).
	 *
	 * @return void
	 */
	public static function resetUser () : void
	{
		static::$userId = null;
	}

}
