<?php

namespace JLGR\FriendInvitations\Livewire;

use Filament\Schemas\Schema;
use Filament\Schemas\Components\Grid;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use JLGR\FriendInvitations\Contracts\InvitesFriend;
use Filament\Forms\Components;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class FriendInvitationForm extends Component implements HasForms
{
	use InteractsWithForms;

	/**
	 * The form's state.
	 * 
	 * @var array
	 */
	public ?array $data = [];

	/**
	 * Flag indicating the visibility of the result message.
	 */
	public $messageSent = false;

	/**
	 * Result object.
	 * 
	 * @var FriendInvitationResult
	 */
	protected $result;

	/**
	 * Mount the component.
	 * 
	 * @return void
	 */
	public function mount ()
	{
		$this->form->fill([
			'text' => $this->getDefaultText()
		]);
	}
	
	/**
	 * Submit the form.
	 * 
	 * @return void
	 */
	public function submitForm (InvitesFriend $processor) : void
	{
		$this->form->validate();

		$this->result = $processor->inviteFriend($this->data);

		$this->finishProcess();
	}
	
	/**
     * Form builder.
     *
     * @param \Filament\Schemas\Schema $schema
     *
     * @return \Filament\Schemas\Schema
     */
    public function form (Schema $schema) : Schema
	{
		return $schema
			->components([
				Grid::make(12)
					->schema([
						TextInput::make('friend_name')
							->label(__('Name of your friend'))
							->columnSpan(8)
							->required()
							->maxLength(255),

						TextInput::make('friend_email')
							->label(__('Your friend\'s e-mail address'))
							->columnSpan(8)
							->prefixIcon('heroicon-m-at-symbol')
							->required()
							->email()
							->maxLength(255),
					]),
				Grid::make(12)
					->schema([
						RichEditor::make('text')
							->label(__('Message to your friend'))
							->columnSpanFull()
							->toolbarButtons([
								'bold',
								'bulletList',
								'h2',
								'h3',
								'italic',
								'link',
								'orderedList',
								'underline',
							])
							->default(static::getDefaultText())
							->required()
							->helperText(__('If you change the invitation text, it will be saved as a template on your profile.'))
					])
			])
			->statePath('data');
	}

	/**
	 * Get default text in case the user hasn't set a invitation text.
	 * 
	 * @return string
	 */
	protected static function getDefaultText ()
	{
		$defaultText = '
			<p>Hi @friend,</p>
			<p>I use @app for sharing my flights with friends and family. Since you are one of my flying friends, I would like to invite you to join my circle of aviators on @app.</p>
			<p>Regards, @name</p>
		';

		return Auth::user()->settings['invitation-email'] ?? $defaultText;
	}

	/**
	 * Finalization of the process.
	 * 
	 * @return void
	 */
	protected function finishProcess ()
	{
		$defaultText = $this->data['text'];

		$this->reset('data');
		$this->data['text'] = $defaultText;

		$this->messageSent = true;
	}


	/**
	 * Render the view.
	 */
	public function render ()
	{
		return view('friend-invitations::livewire.friend-invitation-form.default');
	}

}
