<?php

namespace JLGR\FriendInvitations\Jobs;

use JLGR\FriendInvitations\Mail\FriendInvitationSent;
use JLGR\FriendInvitations\Mail\FriendInvitationSentToAdmin;
use JLGR\FriendInvitations\Models\FriendInvitation;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;

class SendInvitationNotification implements ShouldQueue
{
	use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

	/**
	 * Invitation id.
	 * 
	 * @var integer
	 */
	protected $friendInvitationId;

	/**
	 * Flag to indicate that a notification should be sent to the admin.
	 * 
	 * @var boolean
	 */
	protected $includeAdmin = false;

	/**
	 * Create a new job instance.
	 * 
	 * @param  integer  $friendInvitationId
	 * @param  boolean  $includeAdmin
	 * 
	 * @return void
	 */
	public function __construct ($friendInvitationId, $includeAdmin = false)
	{
		$this->friendInvitationId = $friendInvitationId;
		$this->includeAdmin = $includeAdmin;

		$this->onQueue('social-interaction');
	}

	/**
	 * Execute the job.
	 * Will be processed on the worker server.
	 * 
	 * @return void
	 */
	public function handle () : void
	{
		// Get friend invitation. Abort if not found.
		if (($invitation = $this->getFriendInvitation()) == null) {
			return;
		}

		// Send mail to the friend.
		Mail::to($invitation->friend_email)
			->send(new FriendInvitationSent($invitation));

		// Send mail to admin.
		if ($this->includeAdmin == true)
		{
			$admin = config('mail.identities.admin.address');
			$identifier = config('mail.identifiers.friend-invitations') ?? null;

			Mail::to($admin)
				->send(new FriendInvitationSentToAdmin($invitation, $identifier));
		}
	}

	/**
     * Try to find the given friend invitation.
     *
     * @return FriendInvitation|null
     */
    protected function getFriendInvitation ()
	{
		try
		{
			return FriendInvitation::query()
				->with([
					'user'
				])
				->whereId($this->friendInvitationId)
				->firstOrFail();
		}
		catch (ModelNotFoundException $exception)
		{
			return null;
		}
	}

}
