<?php

namespace JLGR\FriendInvitations\Actions;

use JLGR\FriendInvitations\Contracts\InvitesFriend;
use JLGR\FriendInvitations\Jobs\SendInvitationNotification;
use JLGR\FriendInvitations\Models\FriendInvitation;
use JLGR\FriendInvitations\Results\InviteFriendResult;
use JLGR\FriendInvitations\Services\InvitationService;
use Illuminate\Support\Facades\Auth;

class InviteFriend implements InvitesFriend
{

	/**
	 * Invitation service.
	 * 
	 * @var 
	 */
    protected $invitationService;

    /**
     * Create a new action class instance.
     *
     * @param InvitationService $invitationService
     */
    public function __construct (InvitationService $invitationService)
    {
        $this->invitationService = $invitationService;
    }

	/**
	 * Store invitation.
	 *
	 * @param  array  $input
	 * 
	 * @return InviteFriendResult
	 */
	public function inviteFriend (array $input) : InviteFriendResult
	{
		// Check if the invitation exists in the past year.
		if ($this->invitationService->invitationExistsWithinYear($input['friend_email']))
		{
            return new InviteFriendResult(
				success: false,
				message: 'Invitation already sent within the past year.'
			);
        }

		// Store invitation.
		$invitation = FriendInvitation::create([
			'user_id' => Auth::user()->id,
			'status' => 'new',
			'friend_name' => trim($input['friend_name']),
			'friend_email' => trim($input['friend_email']),
			'text' => $this->invitationService->formatEmailText($input),
		]);

		// Store the template on the user's settings.
		$this->invitationService->storeTemplate($input);

		// Notify invited friend.
		SendInvitationNotification::dispatch($invitation->id, includeAdmin: true);

		// All done.
		return new InviteFriendResult(
			success: true,
			message: 'Thanks, your invitation has been sent!'
		);
	}

}
