<?php

namespace JLGR\Followables\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class FollowedNotification extends Notification implements ShouldQueue
{
	use Queueable;
 
	/**
	 * The follower.
	 *
	 * @var \AviatorsEcho\Models\User
	 */
	protected $follower;
 
	/**
	 * Flag indicating the follow is pending.
	 *
	 * @var boolean
	 */
	protected $isRequested;

	/**
	 * Create a new notification instance.
	 *
	 * @return void
	 */
	public function __construct ($follower, $isRequested)
	{
		$this->follower = $follower;
		$this->isRequested = $isRequested;
	}

	/**
	 * Get the notification's delivery channels.
	 *
	 * @param  mixed  $notifiable
	 * 
	 * @return array
	 */
	public function via ($notifiable)
	{
		return [ 'mail' ];
	}

	/**
	 * Get the mail representation of the notification.
	 *
	 * @param  mixed  $notifiable
	 * 
	 * @return \Illuminate\Notifications\Messages\MailMessage
	 */
	public function toMail ($notifiable)
	{
		return (new MailMessage)
			->from(config('mail.identities.app.address'), config('mail.identities.app.address'))
			->subject($this->getSubject())
			->markdown($this->getEmailTemplate(), [
				'follower' => $this->follower->name,
				'followerId' => $this->follower->id,
			]);
	}

	/**
	 * Get the array representation of the notification.
	 *
	 * @param  mixed  $notifiable
	 * 
	 * @return array
	 */
	public function toArray ($notifiable)
	{
		return [];
	}
	
	/**
	 * Determine which queues should be used for each notification channel.
	 *
	 * @return array
	 */
	public function viaQueues ()
	{
		return [
			'mail' => 'mail-queue',
		];
	}

	/**
	 * Determine the correct email template for this notification.
	 * 
	 * @return string
	 */
	protected function getEmailTemplate ()
	{
		$emailTemplate = $this->isRequested
			? 'follow-request-notification'
			: 'followed-notification';
		
		return 'packages.following.emails.' . $emailTemplate;
	}

	/**
	 * Format subject.
	 * 
	 * @return string
	 */
	protected function getSubject ()
	{
		return $this->isRequested
			? __(':follower has requested to follow you!', [ 'follower' => $this->follower->name ])
			: __(':follower is now following you!', [ 'follower' => $this->follower->name ]);
	}

}
