<?php

namespace JLGR\Followables\Models;

use AviatorsEcho\Models\User;
use JLGR\Followables\Enums\FollowStatus;
use JLGR\Followables\Events\Followed;
use JLGR\Followables\Events\Unfollowed;
use Illuminate\Database\Eloquent\Model;

class Followable extends Model
{

	/**
	 * The table associated with the model.
	 * 
	 * @var string
	 */
	protected $table = 'followables';

	/**
	 * The attributes that are protected from mass assignment. An empty
	 * array means all fields are mass assignable.
	 * 
	 * @var array
	 */
	protected $guarded = [];

	/**
	 * The event map for the model.
	 *
	 * @var array
	 */
	protected $dispatchesEvents = [
		'created' => Followed::class,
		'deleted' => Unfollowed::class,
	];

	/**
	 * Useful local query scopes for filtering on status.
	 *
	 * @param  \Illuminate\Database\Eloquent\Builder $query
	 *
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function scopeWhereAccepted ($query) { return $query->where('status', FollowStatus::FOLLOWING); }

	/**
	 * Get the followable model that the follow belongs to.
	 */
	public function followable ()
	{
		return $this->morphTo();
	}

	/**
	 * Get the followable model that the follow belongs to.
	 */
	public function following ()
	{
		return $this->belongsTo(User::class, 'followable_id');
	}

	/**
	 * Get the follower that the follow belongs to.
	 */
	public function follower ()
	{
		return $this->belongsTo(User::class, 'follower_id');
	}

	/**
	 * Accept follow request.
	 * 
	 * @return void
	 */
	public function accept ()
	{
		$this->update([
			'status' => FollowStatus::FOLLOWING,
			'accepted_at' => now()->toDateTimeString(),
			'blocked_at' => null
		]);
	}

	/**
	 * Accept follow request.
	 * 
	 * @return void
	 */
	public function reject ()
	{
		$this->delete();
	}

	/**
	 * Accept follow request.
	 * 
	 * @return void
	 */
	public function rejectAndBlock ()
	{
		$this->update([
			'status' => FollowStatus::BLOCKED,
			'accepted_at' => null,
			'blocked_at' => now()->toDateTimeString(),
		]);
	}

	/**
	 * Check if the following is requested due to private settings.
	 * 
	 * @return boolean
	 */
	public function isRequested ()
	{
		return $this->status == FollowStatus::REQUESTED;
	}

}
