<?php

namespace JLGR\Followables\Livewire;

use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\{ Computed, On };
use Livewire\Component;

class FollowingButton extends Component
{

	/**
	 * The given user / aviator.
	 * 
	 * @var \AviatorsEcho\Models\User
	 */
	public $aviator;

	/**
	 * The user viewing the given user/ aviator.
	 * 
	 * @var \AviatorsEcho\Models\User
	 */
	public $viewingUser;

	/**
	 * View the button will be used on.
	 * 
	 * @var boolean
	 */
	public $view = 'aviators-profile';

	/**
	 * Button label.
	 * 
	 * @var boolean
	 */
	public $buttonLabel;
	
	/**
	 * Mount the component.
	 * 
	 * @return void
	 */
	public function mount ($aviator, $view = 'aviators-profile') : void
	{
		$this->aviator = $aviator;
		$this->view = $view;

		$this->viewingUser = Auth::user();
		$this->setButtonLabel();
	}

	/**
	 * Flag indicating the authenticated user has requested to follow the given user.
	 * 
	 * @return bool
	 */
	#[ Computed, On('follow-action') ]
	public function hasRequestedToFollow () : bool
	{
		return $this->viewingUser->hasRequestedToFollow($this->aviator);
	}

	/**
	 * Follow the given user.
	 * 
	 * @return void
	 */
	public function follow ()
	{
		$this->viewingUser->follow($this->aviator);

		$this->dispatch('follow-action');
	}

	/**
	 * Unfollow the given user.
	 * 
	 * @return void
	 */
	public function unfollow ()
	{
		$this->viewingUser->unfollow($this->aviator);

		$this->dispatch('follow-action');
	}

	/**
	 * Unblock the given user.
	 * 
	 * @return void
	 */
	public function unblock ()
	{
		$this->viewingUser->unblock($this->aviator);

		$this->dispatch('follow-action');
	}

	/**
	 * Cancel pending follow request. Alias for unfollowing.
	 * 
	 * @return void
	 */
	public function cancelFollowRequest ()
	{
		$this->unfollow();
	}

	/**
	 * Set button label.
	 * 
	 * @return void
	 */
	protected function setButtonLabel () : void
	{
		$this->buttonLabel = [
			'aviators-profile' => $this->aviator->firstName,
			'aviators-grid' => $this->aviator->firstName,
			'homeport-aviators' => $this->aviator->firstName,
		][$this->view];
	}
	
	/**
	 * Render the view.
	 */
	public function render ()
	{
		return view("followables::livewire.buttons.{$this->view}-button");
	}

}
