<?php

namespace JLGR\Followables;

use JLGR\Followables\Events\{ Followed, Unfollowed };
use JLGR\Followables\Listeners\{ DispatchFollowedNotificationJob, DispatchUnfollowedNotificationJob };
use JLGR\Followables\Livewire as LivewireComponents;
use JLGR\Followables\Models\Followable;
use JLGR\Followables\Observers\FollowableObserver;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Livewire\Livewire;

class FollowablesServiceProvider extends ServiceProvider
{
	
	/**
	 * Bootstrap services.
	 * 
	 * @return void
	 */
	public function boot ()
	{
		// Register events.
        $this->registerEvents();

		// Register observers.
        $this->registerObservers();

		// Register livewire components.
		$this->registerLivewireComponents();

		// Publish the package's configuration file.
		$this->publishConfigFile();

		// // Publish the package's migration file.
		$this->publishMigrations();

		$this->loadViewsFrom(__DIR__ . '/../resources/views', 'followables');
	}

	/**
	 * Register services.
	 * 
	 * @return void
	 */
	public function register ()
	{
		// Merge config file.
		$this->mergeConfigFile();
	}

	/**
	 * Register the model observers.
	 * 
	 * @return void
	 */
	protected function registerEvents ()
	{
		// Listen to the Followed event and send out the followed
		// notification.
		Event::listen(
			Followed::class,
			DispatchFollowedNotificationJob::class,
		);

		// Listen to the Unfollowed event and send out the unfollowed
		// notification.
		Event::listen(
			Unfollowed::class,
			DispatchUnfollowedNotificationJob::class,
		);
	}

	/**
	 * Register the model observers.
	 * 
	 * @return void
	 */
	protected function registerObservers ()
	{
		// This observer is responsible for invalidating cache.
		Followable::observe(FollowableObserver::class);
	}

	/**
	 * Register Livewire components.
	 *
	 * @return void
	 */
	protected function registerLivewireComponents ()
	{
		Livewire::component('followables::following-button', LivewireComponents\FollowingButton::class);
		Livewire::component('followables::private-profile-message', LivewireComponents\PrivateProfileMessage::class);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishConfigFile () : void
	{
		$this->publishes([
			__DIR__.'/../config/followables.php' => config_path('jlgr/followables.php'),
		], 'jlgr-config');
	}

	/**
	 * Merge the package's configuration file.
	 * 
	 * @return void
	 */
	protected function mergeConfigFile () : void
	{
		$this->mergeConfigFrom(
			__DIR__ . '/../config/followables.php',
			'jlgr.followables'
		);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishMigrations () : void
	{
		$this->publishes([
			__DIR__.'/../database/migrations' => database_path('migrations'),
		], 'jlgr-migrations');
	}

}
