<?php

namespace JLGR\Followables\Concerns;

use Illuminate\Database\Eloquent\Collection;
use AviatorsEcho\Models\User;
use JLGR\Followables\Enums\FollowStatus;
use JLGR\Followables\Models\Followable;
use JLGR\Followables\Queries\MutualFollowings;
use JLGR\Followables\Queries\SuggestedAviators;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;

trait HasFollowables
{

	/**
	 * Get all the followings for the user. A following is a user being
	 * followed by the given user.
	 */
	public function followings ()
	{
		return $this->hasMany(Followable::class, 'follower_id');
	}

	/**
	 * Get all the followers for the user. Followers are following the 
	 * current authenticated user.
	 */
	public function followers ()
	{
		return $this->hasMany(Followable::class, 'followable_id');
	}

	/**
     * Suggest aviators to the authenticated user.
     *
     * @param  \AviatorsEcho\Models\User  $user
     *
     * @return Collection
     */
    public function networkSuggestions ()
	{
		return SuggestedAviators::get([
			'userId' => $this->id,
			'number' => 10,
		]);
	}

	/**
     * Get mutual followings with the authenticated user.
     *
     * @return Collection
     */
    public function mutualFollowings ($limit = 10)
	{
		return MutualFollowings::collection([
			'recordId' => $this->id,
			'limit' => $limit,
		]);
	}

	/**
	 * Get the number of followings.
	 * 
	 * @return integer
	 */
	public function followingCount ()
	{
		return $this->followings()->whereAccepted()->count();
	}

	/**
	 * Get the number of followers.
	 * 
	 * @return integer
	 */
	public function followerCount ()
	{
		return $this->followers()->whereAccepted()->count();
	}

	/**
	 * Follow the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return \JLGR\Following\Models\Followable
	 */
	public function follow (User $user)
	{
		// Check if the given user can be followed, or that a follow
		// request has to be made.
		$status = ($user->hasPublicProfile())
			? FollowStatus::FOLLOWING
			: FollowStatus::REQUESTED;

		// Set followable date.
		$date = ($user->hasPublicProfile())
			? Carbon::now()->toDateTimeString()
			: null;

		// Create followable.
		return Followable::create([
			'follower_id' => Auth::id(),
			'followable_id' => $user->id,
			'status' => $status,
			'accepted_at' => $date,
		]);
	}

	/**
	 * Unfollow the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function unfollow (User $user)
	{
		if ($following = $this->getFollowing($user)) {
			$following->delete();
		}
	}

	/**
	 * Cancel follow request. Alias for unfollow.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function cancelFollowRequest (User $user)
	{
		$this->unfollow($user);
	}

	/**
	 * Block the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function block (User $user)
	{
		$this->followings()->updateOrCreate(
			[ 'followable_id' => $user->id ],
			[
				'follower_id' => Auth::id(),
				'status' => FollowStatus::BLOCKED,
				'accepted_at' => now()->toDateTimeString()
			]
		);
	}

	/**
	 * Unblock the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function unblock (User $user)
	{
		if ($following = $this->getFollowing($user)) {
			$following->delete();
		}
	}

	/**
	 * Unfollow the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function unfollowAndBlock (User $user)
	{
		if ($following = $this->getFollowing($user)) {
			$following->rejectAndBlock();
		}
	}

	/**
	 * Unblock the given user and accept follow request.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function unblockAndAccept (User $user)
	{
		if ($following = $this->getFollower($user)) {
			$following->accept();
		}
	}

	/**
	 * Accept a follow request from the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return void
	 */
	public function acceptFollowRequestFrom (User $user)
	{
		if ($following = $this->getFollower($user)) {
			$following->accept();
		}
	}

	/**
	 * Reject a follow request from the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * @return void
	 */
	public function rejectFollowRequestFrom (User $user)
	{
		if ($following = $this->getFollower($user)) {
			$following->reject();
		}
	}

	/**
	 * Reject a follow request from the given user and block that user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * @return void
	 */
	public function rejectAndBlockFollowRequestFrom (User $user)
	{
		if ($following = $this->getFollower($user)) {
			$following->rejectAndBlock();
		}
	}

	/**
	 * Determine if the user is followed by the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * @return bool
	 */
	public function isFollowedBy (User $user)
	{
		return !! $this
			->followers()
			->where('id', $user->id)
			->count();
	}

	/**
	 * Determine if the user is following the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * @return bool
	 */
	public function isFollowing (User $user)
	{
		return !! $this
			->followings()
			->where('followable_id', $user->id)
			->where('status', FollowStatus::FOLLOWING)
			->count();
	}

	/**
	 * Determine if the user is not following the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * @return bool
	 */
	public function isNotFollowing (User $user)
	{
		return !! $this
			->followings()
			->where('followable_id', $user->id)
			->where('status', FollowStatus::FOLLOWING)
			->count() == 0;
	}

	/**
	 * Determine if the current user has requested to follow the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return boolean
	 */
	public function hasRequestedToFollow (User $user) : bool
	{
		return !! $this
			->followings()
			->where('followable_id', $user->id)
			->where('status', FollowStatus::REQUESTED)
			->count();
	}

	/**
	 * Determine if the current user has blocked the given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return bool
	 */
	public function hasBlocked (User $user)
	{
		return !! $this
			->followings()
			->where('followable_id', $user->id)
			->where('status', FollowStatus::BLOCKED)
			->count();
	}

	/**
	 * Determine if the current user is blocked by given user.
	 *
	 * @param  \AviatorsEcho\Models\User  $user
	 * 
	 * @return bool
	 */
	public function isBlockedBy (User $user)
	{
		return !! $this
			->followers()
			->where('follower_id', $user->id)
			->where('status', FollowStatus::BLOCKED)
			->count();
	}

	/**
	 * Try to find the follow request date.
	 * 
	 * @param  User  $user
	 * 
	 * @return string|null
	 */
	public function getFollowRequestDate ($user)
	{
		try
		{
			return Followable::query()
				->where('follower_id', $this->id)
				->where('followable_id', $user->id)
				->where('status', FollowStatus::REQUESTED)
				->firstOrFail()
				->created_at
				->format('d-m-Y');
		}
		catch (ModelNotFoundException $exception)
		{
			return null;
		}
	}

	/**
	 * Try to find following.
	 * 
	 * @param  User  $user
	 * 
	 * @return Followable
	 */
	private function getFollowing ($user)
	{
		return Followable::query()
			->where('follower_id', Auth::id())
			->where('followable_id', $user->id)
			->first();
	}

	/**
	 * Try to find follower.
	 * 
	 * @param  User  $user
	 * 
	 * @return Followable
	 */
	private function getFollower ($user)
	{
		return Followable::query()
			->where('follower_id', $user->id)
			->where('followable_id', Auth::id())
			->first();
	}


}
