<?php

namespace JLGR\Enquiries;

use JLGR\Enquiries\Livewire\AdminTable;
use JLGR\Enquiries\Livewire\EnquiryForm;
use JLGR\Enquiries\Contracts\CreatesEnquiry;
use JLGR\Enquiries\Actions\CreateEnquiry;
use JLGR\Enquiries\Repositories\EnquiryRepository;
use JLGR\Enquiries\Models\Enquiry;
use JLGR\Enquiries\Observers\EnquiryObserver;
use Illuminate\Support\ServiceProvider;
use Livewire\Livewire;

class EnquiriesServiceProvider extends ServiceProvider
{

	/**
	 * Bootstrap services.
	 *
	 * @return void
	 */
	public function boot ()
	{
		// Publish the package's configuration file.
		$this->publishConfigFile();

		// Publish the package's migration file.
		$this->publishMigrations();
		
		// Load views.
		$this->loadViewsFrom(__DIR__.'/../resources/views', 'enquiries');

		// Register Livewire components.
		$this->registerLivewireComponents();

		// Register model observers.
		$this->registerObservers();
	}
	
	/**
	 * Register services.
	 *
	 * @return void
	 */
	public function register ()
	{
		// Register action bindings.
		$this->registerActionBindings();

		// Register package repository.
		$this->registerRepository();

		// Merge config file.
		$this->mergeConfigFile();
	}

	/**
	 * Load custom routes.
	 *
	 * @return void
	 */
	protected function registerLivewireComponents ()
	{
		// Admin table component.
		Livewire::component('jlgr.enquiries.admin-table', AdminTable::class);

		// Bug report form.
		Livewire::component('jlgr.enquiries.enquiry-form', EnquiryForm::class);
	}

	/**
	 * Register action bindings.
	 * 
	 * @return void
	 */
	protected function registerActionBindings ()
	{
		$this->app->singleton(
			CreatesEnquiry::class,
			CreateEnquiry::class
		);
	}

	/**
	 * Register the bug report repository.
	 * 
	 * @return void
	 */
	protected function registerRepository () : void
	{
		$this->app->bind(
			EnquiryRepository::class,
			fn ($app) => new EnquiryRepository
		);
	}

	/**
	 * Register observers.
	 * 
	 * @return void
	 */
	protected function registerObservers () : void
	{
		Enquiry::observe(EnquiryObserver::class);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishConfigFile () : void
	{
		$this->publishes([
			__DIR__.'/../config/enquiries.php' => config_path('jlgr/enquiries.php'),
		], 'jlgr-config');
	}

	/**
	 * Merge the package's configuration file.
	 * 
	 * @return void
	 */
	protected function mergeConfigFile () : void
	{
		$this->mergeConfigFrom(
			__DIR__ . '/../config/enquiries.php',
			'jlgr.enquiries'
		);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishMigrations () : void
	{
		$this->publishes([
			__DIR__.'/../database/migrations' => database_path('migrations'),
		], 'jlgr-migrations');
	}

}
