<?php

namespace JLGR\EmailCollector;

use JLGR\EmailCollector\Livewire\CollectEmailForm;
use JLGR\EmailCollector\Contracts\StoresEmailAddress;
use JLGR\EmailCollector\Actions\StoreEmailAddress;
use JLGR\EmailCollector\Repositories\EmailCollectorRepository;
use JLGR\EmailCollector\Models\CollectedEmail;
use JLGR\EmailCollector\Observers\CollectedEmailObserver;
use Illuminate\Support\ServiceProvider;
use Livewire\Livewire;

class EmailCollectorServiceProvider extends ServiceProvider
{

	/**
	 * Bootstrap services.
	 *
	 * @return void
	 */
	public function boot ()
	{
		// Publish the package's configuration file.
		$this->publishConfigFile();

		// Publish the package's migration file.
		$this->publishMigrations();
		
		// Load views.
		$this->loadViewsFrom(__DIR__.'/../resources/views', 'email-collector');

		// Register Livewire components.
		$this->registerLivewireComponents();

		// Register model observers.
		$this->registerObservers();
	}
	
	/**
	 * Register services.
	 *
	 * @return void
	 */
	public function register ()
	{
		// Register action bindings.
		$this->registerActionBindings();

		// Register package repository.
		$this->registerRepository();

		// Merge config file.
		$this->mergeConfigFile();
	}

	/**
	 * Load custom routes.
	 *
	 * @return void
	 */
	protected function registerLivewireComponents ()
	{
		// Collect e-mail form.
		Livewire::component('jlgr.email-collector.collect-email-form', CollectEmailForm::class);
	}

	/**
	 * Register action bindings.
	 * 
	 * @return void
	 */
	protected function registerActionBindings ()
	{
		$this->app->singleton(
			StoresEmailAddress::class,
			StoreEmailAddress::class
		);
	}

	/**
	 * Register the bug report repository.
	 * 
	 * @return void
	 */
	protected function registerRepository () : void
	{
		$this->app->bind(
			EmailCollectorRepository::class,
			fn ($app) => new EmailCollectorRepository
		);
	}

	/**
	 * Register observers.
	 * 
	 * @return void
	 */
	protected function registerObservers () : void
	{
		CollectedEmail::observe(CollectedEmailObserver::class);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishConfigFile () : void
	{
		$this->publishes([
			__DIR__ . '/../config/email-collector.php' => config_path('jlgr/email-collector.php'),
		], 'jlgr-config');
	}

	/**
	 * Merge the package's configuration file.
	 * 
	 * @return void
	 */
	protected function mergeConfigFile () : void
	{
		$this->mergeConfigFrom(
			__DIR__ . '/../config/email-collector.php',
			'jlgr.email-collector'
		);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishMigrations () : void
	{
		$this->publishes([
			__DIR__.'/../database/migrations' => database_path('migrations'),
		], 'jlgr-migrations');
	}

}
