<?php

namespace JLGR\Comments\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\{ BelongsTo, HasMany, MorphTo };
use JLGR\Kernel\Models\BaseModel;

class Comment extends BaseModel
{

	/**
	 * The table associated with the model.
	 * 
	 * @var string
	 */
	protected $table = 'comments';

	/**
	 * The attributes that are protected from mass assignment.
	 *
	 * @var array<int, string>
	 */
	protected $guarded = [];

	/**
	 * Get the attributes that should be cast.
	 *
	 * @return array<string, string>
	 */
	protected function casts () : array
	{
		return [
			'approved_at' => 'datetime',
		];
	}

	/**
	 * Scope a query to only include approved comments.
	 * 
	 * @param  Builder  $builder
	 * 
	 * @return Builder
	 */
	public function scopeApproved (Builder $query) : Builder
	{
		return $query->whereNotNull('approved_at');
	}

	/**
	 * Check if the comment is approved.
	 * 
	 * @return boolean
	 */
	public function isApproved () : bool
	{
		return !! $this->approved_at;
	}

	/**
	 * Get the parent commentable model (post, article, etc).
	 */
	public function commentable () : MorphTo
	{
		return $this->morphTo();
	}

	/**
	 * Get the user that owns the comment.
	 */
	public function user () : BelongsTo
	{
		return $this->belongsTo(config('jlgr.comments.user_model'));
	}

	/**
	 * Get the parent comment (for nested comments).
	 */
	public function parent () : BelongsTo
	{
		return $this->belongsTo(self::class, 'parent_id');
	}

	/**
	 * Get child comments.
	 */
	public function children () : HasMany
	{
		return $this->hasMany(self::class, 'parent_id');
	}

	/**
	 * Get all reports associated with this comment.
	 */
	public function reports () : HasMany
	{
		return $this->hasMany(CommentReport::class);
	}

	/**
	 * Check if the comment is reported by the given user.
	 *
	 * @param  object  $user
	 * 
	 * @return bool
	 */
	public function isReportedBy (object $user) : bool
	{
		return $this->reports()->where('user_id', $user->id)->exists();
	}

	/**
	 * Count how many times this comment has been reported.
	 *
	 * @return int
	 */
	public function reportCount () : int
	{
		return $this->reports()->count();
	}

	/**
	 * Helper to retrieve the depth of the current comment.
	 * 
	 * @return integer
	 */
	public function depth () : int
	{
		// 0 = top-level
		// 1 = reply to top
		// 2 = reply to reply
		$depth = 0;
		$current = $this;

		while ($current->parent) {
			$depth++;
			$current = $current->parent;
		}

		return $depth;
	}

}
