<?php

namespace JLGR\Comments\Livewire;

use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use JLGR\Comments\Contracts\SubmitsComment;
use JLGR\Comments\Models\Comment;
use Livewire\Component;

class CommentForm extends Component implements HasForms
{
	use InteractsWithForms;

	/**
	 * The commentable model.
	 * 
	 * @var mixed
	 */
	public mixed $commentable;

	/**
	 * Optional parent comment (for replies).
	 *
	 * @var int|null
	 */
	public ?int $parentId = null;

	/**
	 * Number of rows for the comment form input.
	 *
	 * @var int
	 */
	public int $rows = 3;

	/**
	 * The component's state.
	 * 
	 * @var array
	 */
	 public ?array $data = [];

	/**
	 * Component properties.
	 * 
	 * @var mixed
	 */
	public string $template = 'default';

	/**
	 * Mount the component with the commentable model.
	 */
	public function mount (mixed $commentable, ?int $parentId = null, int $rows = 3) : void
	{
		$this->commentable = $commentable;
		$this->rows = $rows;

		if ($parentId) {
			$parent = Comment::find($parentId);

			// Allow only replies up to depth 2 (3 levels)
			if (! $parent || $parent->depth() >= 2) {
				$parentId = null;
			}
		}

		$this->parentId = $parentId;

		$this->form->fill([
			'parent_id' => $this->parentId,
		]);
	}
	
	/**
	 * Build the form.
	 * 
	 * @param \Filament\Forms\Form
	 * 
	 * @return \Filament\Forms\Form
	 */
	public function form (Form $form) : Form
	{
		return $form
			->schema([
				Textarea::make('body')
					->hiddenLabel()
					->required()
					->maxLength(1000)
					->rows($this->rows)
					->placeholder(__('Write something nice...'))
					->columnSpanFull()
					->visible($this->template == 'default'),
				TextInput::make('body')
					->hiddenLabel()
					->required()
					->maxLength(1000)
					->placeholder(__('Write something nice and hit enter...'))
					->columnSpanFull()
					->visible($this->template == 'compact')
			])
			->statePath('data');
	}

	/**
	 * Submit the form and create the comment.
	 * 
	 * @param  SubmitsComment  $commentProcessor
	 * 
	 * @return void
	 */
	public function processForm (SubmitsComment $commentProcessor) : void
	{
		$commentProcessor->handle($this->commentable, $this->data);

		$this->reset('data');
		$this->form->fill(); 	// Clear field

		$this->dispatch('comment-submitted');
	}

	/**
	 * Render the component.
	 */
	public function render ()
	{
		return view("comments::livewire.comment-form.{$this->template}");
	}

}
