<?php

namespace JLGR\Comments\Concerns;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Carbon;
use JLGR\Comments\Models\Comment;
use JLGR\Comments\Models\CommentStatistic;

trait HasComments
{

	/**
	 * Get all comments for the model.
	 */
	public function comments () : MorphMany
	{
		return $this->morphMany(Comment::class, 'commentable')->latest();
	}

	/**
	 * Get statistics for this model.
	 */
	public function commentStatistics () : MorphOne
	{
		return $this->morphOne(CommentStatistic::class, 'commentable');
	}

	/**
	 * Add a comment to this model.
	 *
	 * @param  string    $body
	 * @param  int       $userId
	 * @param  int|null  $parentId
	 * @return \JLGR\Comments\Models\Comment
	 */
	public function addComment (string $body, int $userId, ?int $parentId = null) : Comment
	{
		$comment = $this->comments()->create([
			'body'        => $body,
			'user_id'     => $userId,
			'parent_id'   => $parentId,
			'approved_at' => config('jlgr.comments.auto_approve') ? Carbon::now() : null,
		]);

		$this->updateCommentStatistics(
			totalDelta: +1,
			toplevelDelta: $parentId ? 0 : +1
		);

		return $comment;
	}

	/**
	 * Remove a comment from this model.
	 *
	 * @param  integer  $commentId
	 * 
	 * @return void
	 */
	public function removeComment (int $commentId) : void
	{
		$comment = $this->comments()->whereId($commentId)->first();

		if ($comment) {
			$comment->delete();

			$this->updateCommentStatistics(
				totalDelta: -1,
				toplevelDelta: $comment->parent_id ? 0 : -1
			);
		}
	}

	/**
	 * Accessor for total comment count.
	 */
	public function commentsCountTotal () : Attribute
	{
		return Attribute::get(
			fn () => $this->commentStatistics?->count_total ?? 0
		);
	}

	/**
	 * Accessor for top-level comment count.
	 */
	public function commentsCountToplevel () : Attribute
	{
		return Attribute::get(
			fn () => $this->commentStatistics?->count_toplevel ?? 0
		);
	}

	/**
	 * Check if the commentable has any comments.
	 * 
	 * @return boolean
	 */
	public function hasComments () : bool
	{
		return ($this->commentStatistics?->count_total ?? 0) > 0;
	}

	/**
	 * Update comment statistics.
	 *
	 * @param  int  $totalDelta
	 * @param  int  $toplevelDelta
	 * 
	 * @return void
	 */
	public function updateCommentStatistics (int $totalDelta, int $toplevelDelta = 0) : void
	{
		$statistic = CommentStatistic::firstOrCreate([
			'commentable_type' => get_class($this),
			'commentable_id'   => $this->id,
		]);

		$newTotal = max(0, ($statistic->count_total ?? 0) + $totalDelta);
		$newToplevel = max(0, ($statistic->count_toplevel ?? 0) + $toplevelDelta);

		$statistic->update([
			'count_total'   => $newTotal,
			'count_toplevel' => $newToplevel,
		]);
	}

}
