<?php

namespace JLGR\Comments\Actions;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use JLGR\Comments\Contracts\SubmitsComment;
use JLGR\Comments\Models\Comment;

class SubmitComment implements SubmitsComment
{

	/**
	 * Submit new comment.
	 *
	 * @param  mixed  $commentable
	 * @param  array  $data
	 * 
	 * @return void
	 */
	public function handle (mixed $commentable, array $data) : void
	{
		// Validate the parent_id and enforce single-level nesting.
		$parentId = $this->enforceNestingLimit($data);

		// Store comment.
		$commentable->addComment(
			body: Str::trim($data['body']),
			userId: Auth::id(),
			parentId: $parentId,
		);
	}

	/**
	 * Enforce maximum nesting depth of 2 (3 levels total).
	 *
	 * Allows:
	 * - top-level comments
	 * - replies to top-level comments
	 * - replies to replies
	 * 
	 * Disallows:
	 * - replies deeper than that
	 *
	 * @param  array  $data
	 * @return int|null  Valid parent_id, or null if not allowed
	 */
	protected function enforceNestingLimit(array $data): ?int
	{
		if (! isset($data['parent_id'])) {
			return null; // top-level.
		}

		$parent = Comment::find($data['parent_id']);

		if (! $parent) {
			return null; // fallback to top-level.
		}

		// Only allow if depth is less than 3 (i.e. 0, 1 or 2).
		if ($parent->depth() >= 3) {
			return null;
		}

		return $parent->id;
	}

}
