<?php

namespace JLGR\Comments\Livewire;

use Illuminate\Database\Eloquent\Collection;
use Livewire\Component;
use Illuminate\Database\Eloquent\Model;
use JLGR\Comments\Contracts\ReportsComment;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;

class CommentsList extends Component
{
	
	/**
	 * The commentable model.
	 */
	public Model $commentable;

	/**
	 * Whether to show unapproved comments.
	 */
	public bool $includeUnapproved = false;

	/**
	 * The ID of the comment currently being replied to.
	 * When set, the reply form will be shown under this comment.
	 *
	 * @var int|null
	 */
	public ?int $replyTo = null;
	
	/**
	 * The ID of the comment currently being reported.
	 * When set, the reply form will be shown under this comment.
	 *
	 * @var int|null
	 */
	public ?int $reportingComment = null;

	/** 
	 * Array to track which comment threads are expanded. 
	 * Keys are comment IDs, values are booleans.
	 * 
	 * @var array<int, bool>
	 */
	public array $expandedReplies = [];

    /**
     * If true, only top-level comments will be displayed.
     * Replies will be excluded from the view.
     *
     * @var bool
     */
    public bool $onlyTopLevel = false;

    /**
     * Maximum number of top-level comments to display.
     * If null, all comments will be shown.
     *
     * @var int|null
     */
    public ?int $limit = null;

	/**
	 * Component properties.
	 * 
	 * @var mixed
	 */
	public string $template = 'default';

	/**
	 * Mount the component.
	 */
	public function mount (Model $commentable) : void
	{
		$this->commentable = $commentable;

		$this->initialiseCollapsibleThreads();
	}

	/**
	 * Load all top-level comments.
	 */
	#[ On('comment-submitted') ]
	public function refresh () : void
	{
		// Triggers computed property to recompute.
		$this->dispatch('$refresh');
	}

	/**
	 * Computed property:
	 * Root comments with children, optionally filtered on approval.
	 * 
	 * @return Collection<int, Comment>
	 */
	#[ Computed() ]
	public function comments () : Collection
	{
		// Start building the query for top-level comments
		$query = $this->commentable
			->comments()
			->whereNull('parent_id')                // Top-level only
			->with([
				'user.media',
				'children.user.media',
				'children.children.user.media',
			]);

		// Filter out unapproved comments if the flag is disabled
		if (! $this->includeUnapproved) {
			$query->whereNotNull('approved_at');
		}

		// Limit the number of top-level comments if specified
		if (! is_null($this->limit)) {
			$query->limit($this->limit);
		}

		return $query->latest()->get();
	}

	/**
	 * Report a comment as inappropriate.
	 *
	 * @param  int  $commentId
	 * 
	 * @return void
	 */
	public function report (int $commentId, ReportsComment $commentReporter): void
	{
		$commentReporter->handle($commentId);

		$this->reportingComment = null;
	}

	/**
	 * Toggle visibility of replies for a given comment.
	 *
	 * @param  int  $commentId
	 * 
	 * @return void
	 */
	public function toggleReplies (int $commentId) : void
	{
		$this->expandedReplies[$commentId] = !($this->expandedReplies[$commentId] ?? false);
	}

	/**
	 * Toggle visibility of the report comment form.
	 *
	 * @param  int  $commentId
	 * 
	 * @return void
	 */
	public function toggleReport (int $commentId) : void
	{
		$this->reportingComment = $this->reportingComment === $commentId
			? null
			: $commentId;
	}

	/**
	 * Initialise collapsible threads. All threads will be collapsed by
	 * default, except for the first top-level comment.
	 * 
	 * @return void
	 */
	public function initialiseCollapsibleThreads () : void
	{
		$topComments = $this->comments();

		// Expand the most recent top-level comment.
		if ($topComments->isNotEmpty()) {
			$this->expandedReplies[$topComments->first()->id] = true;
		}
	}

	/**
	 * Render the component.
	 */
	public function render ()
	{
		return view("comments::livewire.comments-list.{$this->template}", [
			'comments' => $this->comments,
		]);
	}

}
