<?php

namespace JLGR\Checkins\Livewire;

use Filament\Actions\Action;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Support\Enums\ActionSize;
use Illuminate\Database\Eloquent\Collection;
use Livewire\Component;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use JLGR\Checkins\Models\Checkin;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;

class CheckinsList extends Component implements HasForms, HasActions
{
    use InteractsWithActions;
    use InteractsWithForms;
	
	/**
	 * The checkinable model.
	 */
	public Model $checkinable;

    /**
     * Maximum number of checkins to display. If null, all checkins
	 * will be shown.
     *
     * @var int|null
     */
    public ?int $limit = null;

	/**
	 * Component properties.
	 * 
	 * @var mixed
	 */
	public string $template = 'default';

	/**
	 * Mount the component.
	 */
	public function mount (Model $checkinable) : void
	{
		$this->checkinable = $checkinable;
	}

	/**
	 * Load all top-level checkins.
	 */
	#[ On('checked-in') ]
	public function refresh () : void
	{
		$this->dispatch('$refresh');
	}

	/**
	 * Computed property:
	 * Checkins, optionally filtered on approval.
	 * 
	 * @return Collection<int, Checkin>
	 */
	#[ Computed() ]
	public function checkins () : Collection
	{
		// Start building the query checkins
		$query = $this->checkinable->checkins();

		// Include checkins for only the authenticated user.
		$query->where('user_id', Auth::id());

		// Limit the number of checkins if specified
		if (! is_null($this->limit)) {
			$query->limit($this->limit);
		}

		return $query->get();
	}

	/**
	 * Delete selected checkin.
	 * 
	 * @return Action
	 */
	public function deleteAction () : Action
	{
		return Action::make('delete')
			->requiresConfirmation()
			->iconButton()
			->icon('heroicon-o-trash')
			->color('gray')
			->modalHeading('Delete check-in?')
			->modalDescription('Are you sure you\'d like to delete this check-in? This cannot be undone.')
			->modalSubmitActionLabel('Yes, delete it')
			->action(fn (array $arguments)
				=> Checkin::whereId($arguments['checkinId'])->delete());
	}

	/**
	 * Render the component.
	 */
	public function render ()
	{
		return view("checkins::livewire.checkins-list.{$this->template}", [
			'checkins' => $this->checkins,
		]);
	}

}
