<?php

namespace JLGR\Checkins\Concerns;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\{ MorphMany, MorphOne };
use Illuminate\Support\Facades\Auth;
use JLGR\Checkins\Models\Checkin;
use JLGR\Checkins\Models\CheckinStatistic;

trait CanBeCheckedIn
{

	/**
	 * Get all checkins for the model.
	 */
	public function checkins () : MorphMany
	{
		return $this->morphMany(Checkin::class, 'checkinable')->latest('date');
	}

	/**
	 * Get statistics for this model.
	 */
	public function checkinStatistics () : MorphOne
	{
		return $this->morphOne(CheckinStatistic::class, 'checkinable');
	}

	/**
	 * Get the latest checkin.
	 * 
	 * @return Checkin|null
	 */
	public function latestCheckin () : ?Checkin
	{
		return $this->checkins()->latest('date')->first();
	}

	/**
	 * Get the number of checkins for this checkinable.
	 * 
	 * @param  bool  $unique  Return the unique number of checkins.
	 * 
	 * @return integer
	 */
	public function getCheckinsCount (bool $unique = false) : int
	{
		return $unique === true
			? intval($this->checkinStatistics?->count_unique ?? 0)
			: intval($this->checkinStatistics?->count_total ?? 0);
	}
	
	/**
	 * Accessor for returning the total number of checkins for this model.
	 */
	public function totalCheckinsCount () : Attribute
	{
		return Attribute::get(
			fn () => $this->getCheckinsCount(false)
		);
	}
	
	/**
	 * Accessor for returning the unique number of checkins for this model.
	 */
	public function uniqueCheckinsCount () : Attribute
	{
		return Attribute::get(
			fn () => $this->getCheckinsCount(true)
		);
	}

	/**
	 * Check if the checkinable is checked in to by the authenticated user.
	 * 
	 * @return boolean
	 */
	public function isCheckedInByCurrentUser () : bool
	{
		$userId = Auth::id();

		return $userId
			? $this->checkins()->where('user_id', $userId)->exists()
			: false;
	}

	/**
	 * Get the number of checkins for the authenticated user.
	 * 
	 * @return integer
	 */
	public function checkinsByCurrentUserCount () : int
	{
		$userId = Auth::id();

		if (! $userId) {
			return 0;
		}

		return $this->checkins()
			->where('user_id', $userId)
			->count();
	}

	/**
	 * Check if the checkinable has any checkins.
	 * 
	 * @return boolean
	 */
	public function hasCheckins () : bool
	{
		return $this->getCheckinsCount() > 0;
	}

	/**
	 * Update like statistics.
	 * 
	 * @param  integer  $totalDelta
	 * @param  integer  $uniqueDelta
	 * 
	 * @return void
	 */
	public function updateCheckinStatistics (int $totalDelta, int $uniqueDelta) : void
	{
		$statistic = CheckinStatistic::firstOrCreate([
			'checkinable_type' => get_class($this),
			'checkinable_id' => $this->id,
		]);

        $statistic->update([
            'count_total'  => max(0, $statistic->count_total + $totalDelta),
            'count_unique' => max(0, $statistic->count_unique + $uniqueDelta),
        ]);
	}

}
