<?php

namespace JLGR\BugReports;

use JLGR\BugReports\Livewire\AdminTable;
use JLGR\BugReports\Livewire\BugReportForm;
use JLGR\BugReports\Contracts\CreatesBugReport;
use JLGR\BugReports\Actions\CreateBugReport;
use JLGR\BugReports\Repositories\BugReportRepository;
use JLGR\BugReports\Models\BugReport;
use JLGR\BugReports\Observers\BugReportObserver;
use Illuminate\Support\ServiceProvider;
use Livewire\Livewire;

class BugReportsServiceProvider extends ServiceProvider
{

	/**
	 * Bootstrap services.
	 *
	 * @return void
	 */
	public function boot ()
	{
		// Publish the package's configuration file.
		$this->publishConfigFile();

		// Publish the package's migration file.
		$this->publishMigrations();
		
		// Load views.
		$this->loadViewsFrom(__DIR__.'/../resources/views', 'bug-reports');

		// Register Livewire components.
		$this->registerLivewireComponents();

		// Register model observers.
		$this->registerObservers();
	}
	
	/**
	 * Register services.
	 *
	 * @return void
	 */
	public function register ()
	{
		// Register action bindings.
		$this->registerActionBindings();

		// Register package repository.
		$this->registerRepository();
	}

	/**
	 * Load custom routes.
	 *
	 * @return void
	 */
	protected function registerLivewireComponents ()
	{
		Livewire::component('bug-reports::admin-table', AdminTable::class);
		Livewire::component('bug-reports::bug-report-form', BugReportForm::class);
	}

	/**
	 * Register action bindings.
	 * 
	 * @return void
	 */
	protected function registerActionBindings ()
	{
		$this->app->singleton(
			CreatesBugReport::class,
			CreateBugReport::class
		);
	}

	/**
	 * Register the bug report repository.
	 * 
	 * @return void
	 */
	protected function registerRepository () : void
	{
		$this->app->bind(
			BugReportRepository::class,
			fn ($app) => new BugReportRepository
		);
	}

	/**
	 * Register observers.
	 * 
	 * @return void
	 */
	protected function registerObservers () : void
	{
		BugReport::observe(BugReportObserver::class);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishConfigFile () : void
	{
		$this->publishes([
			__DIR__.'/../config/bug-reports.php' => config_path('jlgr/bug-reports.php'),
		], 'jlgr-config');
	}

	/**
	 * Merge the package's configuration file.
	 * 
	 * @return void
	 */
	protected function mergeConfigFile () : void
	{
		$this->mergeConfigFrom(
			__DIR__ . '/../config/bug-reports.php',
			'jlgr.bug-reports'
		);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishMigrations () : void
	{
		$this->publishes([
			__DIR__.'/../database/migrations' => database_path('migrations'),
		], 'jlgr-migrations');
	}

}
