<?php

namespace JLGR\BugReports;

use Illuminate\Support\ServiceProvider;
use Livewire\Livewire;

class BugReportsServiceProvider extends ServiceProvider
{

	/**
	 * Bootstrap services.
	 *
	 * @return void
	 */
	public function boot ()
	{
		// Publish the package's configuration file.
		$this->publishConfigFile();

		// Publish the package's migration file.
		$this->publishMigrations();
		
		// Load views.
		$this->loadViewsFrom(__DIR__.'/../resources/views', 'bug-reports');

		// Register Livewire components.
		$this->registerLivewireComponents();

		// Register model observers.
		$this->registerObservers();
	}
	
	/**
	 * Register services.
	 *
	 * @return void
	 */
	public function register ()
	{
		// Register action bindings.
		$this->registerActionBindings();

		// Register package repository.
		$this->registerRepository();
	}

	/**
	 * Load custom routes.
	 *
	 * @return void
	 */
	protected function registerLivewireComponents ()
	{
		Livewire::component('bug-reports::admin-table', \JLGR\BugReports\Livewire\AdminTable::class);
		Livewire::component('bug-reports::bug-report-form', \JLGR\BugReports\Livewire\BugReportForm::class);
	}

	/**
	 * Register action bindings.
	 * 
	 * @return void
	 */
	protected function registerActionBindings ()
	{
		$this->app->singleton(
			\JLGR\BugReports\Contracts\CreatesBugReport::class,
			\JLGR\BugReports\Actions\CreateBugReport::class
		);
	}

	/**
	 * Register the bug report repository.
	 * 
	 * @return void
	 */
	protected function registerRepository () : void
	{
		$this->app->bind(
			\JLGR\BugReports\Repositories\BugReportRepository::class,
			fn ($app) => new \JLGR\BugReports\Repositories\BugReportRepository
		);
	}

	/**
	 * Register observers.
	 * 
	 * @return void
	 */
	protected function registerObservers () : void
	{
		\JLGR\BugReports\Models\BugReport::observe(\JLGR\BugReports\Observers\BugReportObserver::class);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishConfigFile () : void
	{
		$this->publishes([
			__DIR__.'/../config/bug-reports.php' => config_path('jlgr/bug-reports.php'),
		], 'jlgr-config');
	}

	/**
	 * Merge the package's configuration file.
	 * 
	 * @return void
	 */
	protected function mergeConfigFile () : void
	{
		$this->mergeConfigFrom(
			__DIR__ . '/../config/bug-reports.php',
			'jlgr.bug-reports'
		);
	}

	/**
	 * Publish the package's configuration file.
	 * 
	 * @return void
	 */
	protected function publishMigrations () : void
	{
		$this->publishes([
			__DIR__.'/../database/migrations' => database_path('migrations'),
		], 'jlgr-migrations');
	}

}
